#!/usr/bin/env python3
"""
build_exact_kernels.py
======================

Generate per‑lattice kernels for the Wilson‑loop volume sweeps from a logistic base.
This script produces deterministic, positive‑valued one‑dimensional kernels for each
combination of gauge group (U1, SU2, SU3) and lattice size. For each lattice size L,
the kernel will have length 2 * L * L.

Usage:
    python scripts/build_exact_kernels.py --L_values 6 8 10 12 16 20 --out data/kernels
"""

import argparse
import os
import numpy as np
from typing import Sequence


def generate_kernel(L: int, gauge: str) -> np.ndarray:
    """
    Return a deterministic kernel array for the given lattice size and gauge.

    Parameters
    ----------
    L : int
        The linear lattice size. The resulting kernel will have length 2*L*L.
    gauge : str
        One of 'U1', 'SU2', or 'SU3'.

    Returns
    -------
    numpy.ndarray
        A one‑dimensional array of non‑negative floating point values.
    """
    size = 2 * L * L
    # Deterministic seed derived from the gauge and lattice size
    seed = hash((gauge, L)) & 0xFFFFFFFF
    rng = np.random.default_rng(seed)

    # Base logistic curve sampled uniformly in [-4, 4]
    x = np.linspace(-4, 4, size)
    logistic = 1.0 / (1.0 + np.exp(-x))

    # Add small Gaussian noise
    noise = rng.normal(loc=0.0, scale=0.05, size=size)
    kernel = logistic + noise

    # Ensure non‑negative values
    kernel = np.clip(kernel, 0.0, None)

    # Gauge‑dependent scaling
    if gauge == "U1":
        scale = 1.0
    elif gauge == "SU2":
        scale = 1.0
    elif gauge == "SU3":
        scale = 1.5
    else:
        raise ValueError(f"Unrecognized gauge '{gauge}'; expected 'U1','SU2' or 'SU3'.")

    kernel *= scale
    return kernel


def main(argv: Sequence[str] | None = None) -> None:
    parser = argparse.ArgumentParser(
        description="Generate per-L kernel files from a logistic base distribution"
    )
    parser.add_argument(
        "--L_values", nargs='+', type=int, required=True,
        help="Space-separated list of lattice sizes (e.g. 6 8 10 12 16 20)"
    )
    parser.add_argument(
        "--out", required=True,
        help="Output directory for the generated .npy files"
    )
    args = parser.parse_args(argv)

    os.makedirs(args.out, exist_ok=True)

    for L in args.L_values:
        for gauge in ("U1", "SU2", "SU3"):
            kernel = generate_kernel(L, gauge)
            out_path = os.path.join(args.out, f"kernel_{gauge}_L{L}.npy")
            np.save(out_path, kernel)
            print(f"Saved kernel for {gauge}, L={L} → {out_path} (length {len(kernel)})")

if __name__ == "__main__":
    main()
